#ifdef _DEBUG
#undef _DEBUG
#include <Python.h>
#define _DEBUG
#else
#include <Python.h>
#endif
#include "enet.h"

// #define DO_LOG
#ifdef DO_LOG
#define DEBUGLOG(msg)   printf(msg)
#else
#define DEBUGLOG(msg)
#endif

/*
    :TODO:
    Destruct/finalize module?
*/
static PyTypeObject enet_HostType;
static PyTypeObject enet_PeerType;
static PyTypeObject enet_EventType;

typedef struct {
    PyObject_HEAD
    ENetHost* host;
} enet_HostObject;

typedef struct {
    PyObject_HEAD
    ENetPeer* peer;
} enet_PeerObject;

typedef struct {
    PyObject_HEAD
    ENetEvent event;
} enet_EventObject;

static char enet_doc[] =
    "Contains routines for dealing with the keyboard. All keyboard\n"
    "events can be retreived through the pygame.event module. With the\n"
    "key module, you can get the current state of the keyboard, as\n"
    "well as set the rate of keyboard repeating and lookup names of\n"
    "keysyms.\n";

static PyObject *enet_EventObject_GetAttr(PyObject *self, PyObject *attr_name)
{
    enet_EventObject *event_obj = (enet_EventObject*)self;
    ENetEvent *event = &event_obj->event;
    char *name = PyString_AS_STRING(attr_name);
    PyObject *attr;

    if (0 == strcmp(name, "channel"))
    {
        return PyInt_FromLong((long)event->channelID);
    }
    else if (0 == strcmp(name, "data"))
    {
        if (ENET_EVENT_TYPE_RECEIVE == event->type)
            return PyString_FromStringAndSize(event->packet->data,
                                              (int)event->packet->dataLength);
    }
    else if (0 == strcmp(name, "type"))
    {
        return PyInt_FromLong((long)event->type);
    }
    else if (0 == strcmp(name, "peer"))
    {
        enet_PeerObject* peer_obj = PyObject_New(enet_PeerObject, &enet_PeerType);
        peer_obj->peer = event->peer;
        return (PyObject*)peer_obj;
    }

    attr = PyObject_GenericGetAttr(self, attr_name);
    // :TODO: Should this be DECREF?
    if (NULL != attr)
        Py_INCREF(Py_None);
    return attr;
}

/* Free any external resources here;
 * if the instance owns references to any Python
 * objects, call Py_DECREF() on them here.
 */
static void enet_EventObjectDealloc(PyObject* self)
{
    enet_EventObject* event_obj = (enet_EventObject*)self;
    PyObject_Del(self);
}

static PyTypeObject enet_EventType = {
    /* The ob_type field must be initialized in the module init function
     * to be portable to Windows without using C++. */
    PyObject_HEAD_INIT(NULL)
    0,       /*ob_size*/
    "Event",      /*tp_name*/
    sizeof(enet_EventType),   /*tp_basicsize*/
    0,       /*tp_itemsize*/
    /* methods */
    enet_EventObjectDealloc, /*tp_dealloc*/
    0,       /*tp_print*/
    0, /*tp_getattr*/
    0, /*tp_setattr*/
    0,       /*tp_compare*/
    0,       /*tp_repr*/
    0,       /*tp_as_number*/
    0,       /*tp_as_sequence*/
    0,       /*tp_as_mapping*/
    0,       /*tp_hash*/
    0, // (ternaryfunc)enet_HostType_Call,  /*tp_call*/  // **** FUNCTIONCALL CATCHER
    0,                      /*tp_str*/
    enet_EventObject_GetAttr, /*tp_getattro*/
    PyObject_GenericSetAttr,  /*tp_setattro*/
    0,                      /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT,     /*tp_flags*/
    0,                      /*tp_doc*/
    0,                      /*tp_traverse*/
    0,                      /*tp_clear*/
    0,                      /*tp_richcompare*/
    0,                      /*tp_weaklistoffset*/
    0,                      /*tp_iter*/
    0,                      /*tp_iternext*/
    0,                      /*tp_methods*/
    0,                      /*tp_members*/
    0,                      /*tp_getset*/
    0,                      /*tp_base*/
    0,                      /*tp_dict*/
    0,                      /*tp_descr_get*/
    0,                      /*tp_descr_set*/
    0,                      /*tp_dictoffset*/
    0,                      /*tp_init*/
    0,                      /*tp_alloc*/
    0,                      /*tp_new*/
    0,                      /*tp_free*/
    0,                      /*tp_is_gc*/
};

static char enet_HostObjectNew_doc[] =
    "host host((addr, port), int peercount, int incomingbandwidth, int outgoingbandwidth)\n"
    "host host(int peercount, int incomingbandwidth, int outgoingbandwidth)\n"
    "Clients should use the latter version.\n"
    "\n"
    "Creates a host for communicating with peers. The address parameter\n"
    "specifies the address at which other peers may connect to this host;\n"
    "if the address parameter is NULL, then no peers may connect to the host.\n"
    "The peerCount parameter specifies the numbers of peers that should be\n"
    "allocated for the host; this limits the maximum number of peers that may\n"
    "connect to this host to peerCount. The incomingBandwidth parameter\n"
    "specifies the downstream bandwidth of the host in bytes per second; if\n"
    "the incomingBandwidth parameter is 0, ENet will assume the host has\n"
    "unlimited downstream bandwidth. The outgoingBandwidth parameter\n"
    "specifies the upstream bandwidth of the host in bytes per second; if\n"
    "the outgoingBandwidth parameter is 0, ENet will assume the host has\n"
    "unlimited upstream bandwidth. ENet will strategically drop packets on\n"
    "specific sides of a connection between hosts to ensure the host's\n"
    "bandwidth is not overwhelmed; the bandwidth parameters also determine\n"
    "the window size of a connection which limits the amount of reliable\n"
    "packets that may be in transit at any given time. Returns the host on\n"
    "success and NULL on failure.\n";
static PyObject* enet_HostObjectNew(PyObject *self, PyObject *args, PyObject *kwds)
{
    ENetHost* host;
    enet_HostObject* host_obj;
    ENetAddress address;
    char* addr;
    int port;
    size_t peercount;
    enet_uint32 incomingbandwidth, outgoingbandwidth;

    DEBUGLOG("host_create: enter\n");
    if (0 != PyArg_ParseTuple(args, "(si)iii", &addr, &port, &peercount, &incomingbandwidth, &outgoingbandwidth))
    {
        DEBUGLOG("(si)iii\n");
        enet_address_set_host(&address, addr);
        address.port = port;
        host = enet_host_create(&address, peercount, incomingbandwidth, outgoingbandwidth);
        if (NULL != host)
        {
            host_obj = PyObject_New(enet_HostObject, &enet_HostType);
            host_obj->host = host;
            return (PyObject*)host_obj;
        }

        Py_INCREF(Py_None);
        return Py_None;
    }
    else if (0 != PyArg_ParseTuple(args, "iii", &peercount, &incomingbandwidth, &outgoingbandwidth))
    {
        PyErr_Clear();

        DEBUGLOG("iii\n");
        host = enet_host_create(NULL, peercount, incomingbandwidth, outgoingbandwidth);
        if (NULL != host)
        {
            host_obj = PyObject_New(enet_HostObject, &enet_HostType);
            host_obj->host = host;
            return (PyObject*)host_obj;
        }
        
        Py_INCREF(Py_None);
        return Py_None;
    }

    return NULL;
}

/* Free any external resources here;
 * if the instance owns references to any Python
 * objects, call Py_DECREF() on them here.
 */
static void enet_HostObjectDealloc(PyObject* self)
{
    enet_HostObject* host_obj;
    
    DEBUGLOG("Host Dealloc\n");
    host_obj = (enet_HostObject*)self;
    if (NULL != host_obj->host)
        enet_host_destroy(host_obj->host);

    PyObject_Del(self);
}

static char enet_HostObject_destroy_doc[] =
    "void destroy()\n"
    "Destroys the host and all resources associated with it.\n";
static PyObject* enet_HostObject_destroy(PyObject* self)
{
    DEBUGLOG("Host Destroy\n");

    PyObject_Del(self);
    Py_INCREF(Py_None);
    return Py_None;
}

static char enet_HostObject_connect_doc[] =
    "void connect((string addr, int port), int numchannels)\n"
    "Initiates a connection from the host specified in the host parameter\n"
    "to a foreign host whose internet address is specified by the address\n"
    "parameter. The channelCount parameter specifies the number of channels\n"
    "that should be allocated for communicating with the foreign host.\n"
    "Returns a peer representing the foreign host on success and NULL on\n"
    "failure. The peer returned will have not completed the connection\n"
    "until enet_host_service notifies of an ENET_EVENT_TYPE_CONNECT event\n"
    "for the peer.\n";
static PyObject* enet_HostObject_connect(PyObject* self, PyObject* args)
{
    enet_HostObject* host_obj;
    ENetPeer* peer;
    ENetAddress address;
    char* addr;
    int port;
    size_t channelcount;
    
    DEBUGLOG("enet_HostObject_connect: Entered\n");

    if (0 == PyArg_ParseTuple(args, "(si)i", &addr, &port, &channelcount))
        return NULL;

    enet_address_set_host(&address, addr);
    address.port = port;
    host_obj = (enet_HostObject*)self;
    peer = enet_host_connect(host_obj->host, &address, channelcount);
    if (NULL != peer)
    {
        enet_PeerObject* peer_obj;
        peer_obj = PyObject_New(enet_PeerObject, &enet_PeerType);
        peer_obj->peer = peer;
        DEBUGLOG("enet_HostObject_connect: Success\n");

        return (PyObject*)peer_obj;
    }

    Py_INCREF(Py_None);
    return Py_None;
}

static char enet_HostObject_service_doc[] =
    "result, event service(int timeout)\n"
    "Waits for events on the host specified by the host parameters and\n"
    "shuttles packets between the host and its peers. The event parameter\n"
    "specifies an event structure where event details will be placed if one\n"
    "occurs. The timeout field specifies an amount of time in milliseconds\n"
    "that ENet should wait for events. Returns 1 if an event occured within\n"
    "the specified time limit, 0 if no event occurred within the time limit,\n"
    "and -1 on failure. This function must be called frequently for adequate\n"
    "performance.\n";
static PyObject* enet_HostObject_service(enet_HostObject* self, PyObject* args)
{
    enet_EventObject* event_obj = NULL;
    PyObject* out;
    enet_uint32 timeout;
    int result = -1;

    if (0 == PyArg_ParseTuple(args, "i", &timeout))
        return NULL;

    out = PyTuple_New(2);
    event_obj = PyObject_New(enet_EventObject, &enet_EventType);
    result = enet_host_service(self->host, &event_obj->event, timeout);
    if (1 == result)
    {
        PyTuple_SetItem(out, 1, (PyObject*)event_obj);
    }
    /* Not successful, for whatever reason */
    else
    {
        Py_INCREF(Py_None);
        PyTuple_SetItem(out, 1, Py_None);
        PyObject_Del(event_obj);
    }

    PyTuple_SetItem(out, 0, PyInt_FromLong((long)result));
    
    return out; 
}

static char enet_HostObject_flush_doc[] =
    "void flush()\n"
    "Sends out any queued packets on the host specified in the host\n"
    "parameters to the designated peers. This function need only be used\n"
    "in circumstances where one wishes to send queued packets earlier than\n"
    "in a call to enet_host_service.\n";
static PyObject* enet_HostObject_flush(PyObject* self)//, PyObject* args)
{
    enet_HostObject* host_obj;

    host_obj = (enet_HostObject*)self;
    enet_host_flush(host_obj->host);

    Py_INCREF(Py_None);
    return Py_None;
}

static char enet_HostObject_broadcast_doc[] =
    "void broadcast(int channel, string data, int flags)\n"
    "Queues a packet to be sent to all peers on the host specified\n"
    "in the host parameter over the channel number identified by\n"
    "the channel parameter.\n";
static PyObject* enet_HostObject_broadcast(PyObject* self, PyObject* args)
{
    enet_HostObject* host_obj;
    ENetPacket* packet;
    char* data;
    int channel, flags; 

    DEBUGLOG("enet_HostObject_broadcast: Enter\n");
    if (0 != PyArg_ParseTuple(args, "isi", &channel, &data, &flags))
    {
        packet = enet_packet_create(data, strlen(data), flags);
        if (NULL != packet)
        {
            host_obj = (enet_HostObject*)self;
            enet_host_broadcast(host_obj->host, channel, packet);
            DEBUGLOG("enet_HostObject_broadcast: Success\n");
        }

        Py_INCREF(Py_None);
        return Py_None;
    }
    else if (0 != PyArg_ParseTuple(args, "is", &channel, &data))
    {
        PyErr_Clear();

        packet = enet_packet_create(data, strlen(data), 0);
        if (NULL != packet)
        {
            host_obj = (enet_HostObject*)self;
            enet_host_broadcast(host_obj->host, channel, packet);
            DEBUGLOG("enet_HostObject_broadcast: Success\n");
        }

        Py_INCREF(Py_None);
        return Py_None;
    }

    return NULL;
}

static char enet_HostObject_bandwidth_limit_doc[] =
    "void bandwidth(int incomingbandwidth, int outgoingbandwidth)\n"
    "Adjusts the bandwidth limits of the host specified in the host\n"
    "parameter. The incomingBandwidth and outgoingBandwidth parameters are\n"
    "as specified in a call to enet_host_create.\n";
static PyObject* enet_HostObject_bandwidth_limit(enet_HostObject* self, PyObject* args)
{
    int incoming, outgoing;

    if (0 == PyArg_ParseTuple(args, "ii", &incoming, &outgoing))
        return NULL;

    enet_host_bandwidth_limit(self->host, incoming, outgoing);

    Py_INCREF(Py_None);
    return Py_None;
}

static PyMethodDef enet_HostType_methods[] = {
    {"destroy", (PyCFunction)enet_HostObject_destroy, METH_NOARGS, enet_HostObject_destroy_doc},
    {"connect", (PyCFunction)enet_HostObject_connect, METH_VARARGS, enet_HostObject_connect_doc},
    {"service", (PyCFunction)enet_HostObject_service, METH_VARARGS, enet_HostObject_service_doc},
    {"flush", (PyCFunction)enet_HostObject_flush, METH_NOARGS, enet_HostObject_flush_doc},
    {"broadcast", (PyCFunction)enet_HostObject_broadcast, METH_VARARGS, enet_HostObject_broadcast_doc},
    {"bandwidth_limit", (PyCFunction)enet_HostObject_bandwidth_limit, METH_VARARGS, enet_HostObject_bandwidth_limit_doc},
    {NULL, NULL, 0, NULL} /* sentinel */
};

static PyTypeObject enet_HostType = {
    /* The ob_type field must be initialized in the module init function
     * to be portable to Windows without using C++. */
    PyObject_HEAD_INIT(NULL)
    0,       /*ob_size*/
    "Host",      /*tp_name*/
    sizeof(enet_HostType),   /*tp_basicsize*/
    0,       /*tp_itemsize*/
    /* methods */
    enet_HostObjectDealloc, /*tp_dealloc*/
    0,       /*tp_print*/
    0, /*tp_getattr*/
    0, /*tp_setattr*/
    0,       /*tp_compare*/
    0,       /*tp_repr*/
    0,       /*tp_as_number*/
    0,       /*tp_as_sequence*/
    0,       /*tp_as_mapping*/
    0,       /*tp_hash*/
    0, // (ternaryfunc)enet_HostType_Call,  /*tp_call*/  // **** FUNCTIONCALL CATCHER
    0,                      /*tp_str*/
    PyObject_GenericGetAttr, /*tp_getattro*/
    PyObject_GenericSetAttr, /*tp_setattro*/
    0,                      /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT,     /*tp_flags*/
    0,                      /*tp_doc*/
    0,                      /*tp_traverse*/
    0,                      /*tp_clear*/
    0,                      /*tp_richcompare*/
    0,                      /*tp_weaklistoffset*/
    0,                      /*tp_iter*/
    0,                      /*tp_iternext*/
    enet_HostType_methods,  /*tp_methods*/
    0,                      /*tp_members*/
    0,                      /*tp_getset*/
    0,                      /*tp_base*/
    0,                      /*tp_dict*/
    0,                      /*tp_descr_get*/
    0,                      /*tp_descr_set*/
    0,                      /*tp_dictoffset*/
    0,                      /*tp_init*/
    0,                      /*tp_alloc*/
    0,                      /*tp_new*/
    0,                      /*tp_free*/
    0,                      /*tp_is_gc*/
};

/* Free any external resources here;
 * if the instance owns references to any Python
 * objects, call Py_DECREF() on them here.
 */
static void enet_PeerObjectDealloc(PyObject* self)
{
    enet_PeerObject* peer_obj = (enet_PeerObject*)self;
    
    DEBUGLOG("Peer Dealloc\n");
    enet_peer_disconnect(peer_obj->peer);
    PyObject_Del(self);
}

static char enet_PeerObject_send_doc[] =
    "void send(int channel, string data, int flags)\n"
    "Queues a packet to be sent to the peer specified by the peer\n"
    "parameter over the channel number identified by the channelID\n"
    "parameter. Returns 0 on success and -1 on failure.\n";
static PyObject* enet_PeerObject_send(enet_PeerObject* self, PyObject* args)
{
    ENetPacket* packet;
    char* data;
    int channel, flags; 

    if (0 == PyArg_ParseTuple(args, "isi", &channel, &data, &flags))
        return NULL;

    packet = enet_packet_create(data, strlen(data), flags);
    if (NULL != packet)
    {
        if (0 == enet_peer_send(self->peer, channel, packet))
            return PyInt_FromLong(1);
    }

    return PyInt_FromLong(0);
}

static char enet_PeerObject_receive_doc[] =
    "string receive(int channel)\n"
    "Attempts to dequeue any incoming queued packets on the peer\n"
    "specified by the peer parameter on the channel number identified\n"
    "by the channel parameter. Returns a packet if one is available\n"
    "and NULL if there are no available incoming queued packets.\n";
static PyObject* enet_PeerObject_receive(PyObject* self, PyObject* args)
{
    enet_PeerObject* peer_obj;
    ENetPacket* packet;
    PyObject* data;
    int channel; 

    if (0 == PyArg_ParseTuple(args, "i", &channel))
        return NULL;

    peer_obj = (enet_PeerObject*)self;
    packet = enet_peer_receive(peer_obj->peer, channel);
    if (NULL != packet)
    {
        DEBUGLOG("enet_PeerObject_receive: Success");
        data = PyString_FromStringAndSize(packet->data, (int)packet->dataLength);
        enet_packet_destroy(packet);
        return data;
    }

    DEBUGLOG("enet_PeerObject_receive: Exit");
    Py_INCREF(Py_None);
    return Py_None;
}

static char enet_PeerObject_ping_doc[] =
    "void ping()\n"
    "Sends a ping request to the peer specified by the peer parameter.\n"
    "Ping requests factor into the mean round trip time as designated by\n"
    "the roundTripTime field in the ENetPeer? structure. ENet automatically\n"
    "pings all connected peer at an interval, however, this function may be\n"
    "called to ensure more frequent ping requests.\n";
static PyObject* enet_PeerObject_ping(PyObject* self, PyObject* args)
{
    enet_PeerObject* peer_obj;

    peer_obj = (enet_PeerObject*)self;
    enet_peer_ping(peer_obj->peer);

    Py_INCREF(Py_None);
    return Py_None;
}

static char enet_PeerObject_reset_doc[] =
    "void reset()\n"
    "Forcefully disconnects the peer specified by the peer parameter.\n"
    "The foreign host represented by the peer is not notified of the\n"
    "disconnection and so will timeout on its connection to the local host.\n";
static PyObject* enet_PeerObject_reset(enet_PeerObject* self, PyObject* args)
{
    enet_peer_reset(self->peer);
    Py_INCREF(Py_None);
    return Py_None;
}

static char enet_PeerObject_disconnect_doc[] =
    "void disconnect()\n"
    "Request a disconnection from the peer specified by the peer parameter.\n"
    "An ENET_EVENT_DISCONNECT event will be generated by enet_host_service\n"
    "once the disconnection is complete.\n";
static PyObject* enet_PeerObject_disconnect(enet_PeerObject* self)
{
    enet_peer_disconnect(self->peer);
    DEBUGLOG("Disconnect\n");
    Py_INCREF(Py_None);
    return Py_None;
}

static char enet_PeerObject_throttle_configure_doc[] =
    "void configure(int interval, int accel, int decel)\n"
    "Configures throttle parameter for the peer specified by the peer\n"
    "parameter. Unreliable packets are dropped by ENet in response to the\n"
    "varying conditions of the internet connection to the peer. The throttle\n"
    "represents a probability that an unreliable packet should not be dropped\n"
    "and thus sent by ENet to the peer. The lowest mean round trip time from\n"
    "the sending of a reliable packet to the receipt of its acknowledgement\n"
    "is measured over an amount of time specified by the interval parameter\n"
    "in milliseconds; the constant ENET_PEER_PACKET_THROTTLE_INTERVAL is the\n"
    "default value for this parameter. If a measured round trip time happens\n"
    "to be signifigantly less than the mean round trip time measured over the\n"
    "interval, then the throttle probability is increased to allow more\n"
    "traffic by an amount specified in the acceleration parameter which is\n"
    "in ratio to the ENET_PEER_PACKET_THROTTLE_SCALE constant. If a measured\n"
    "round trip time happens to be signifigantly greater than the mean round\n"
    "trip time measured over the interval, then the throttle probability is\n"
    "decreased to limit traffic by an amount specified in the deceleration\n"
    "parameter which is in ratio to the ENET_PEER_PACKET_THROTTLE_SCALE\n"
    "constant. When the throttle has a value of\n"
    "ENET_PEER_PACKET_THROTTLE_SCALE, no unreliable packets are dropped by\n"
    "ENET, and so 100% of all unreliable packets will be sent. When the\n"
    "throttle has a value of 0, all unreliable packets are dropped by ENet,\n"
    "and so 0% of all unreliable packets will be sent. Intermediate values\n"
    "for the throttle represent intermediate probabilities between 0% and\n"
    "100% of unreliable packets being sent. The bandwidth limits of the\n"
    "local and foreign host are taken into account to determine a sensible\n"
    "limit for the throttle probability above which it should not raise\n"
    "even in the best of conditions.\n";
static PyObject* enet_PeerObject_throttle_configure(enet_PeerObject* self, PyObject* args)
{
    int interval, accel, decel;

    if (0 == PyArg_ParseTuple(args, "iii", &interval, &accel, &decel))
        return NULL;

    enet_peer_throttle_configure(self->peer, interval, accel, decel);
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *enet_PeerObject_GetAttr(enet_PeerObject *self, PyObject *attr_name)
{
    ENetPeer* peer = self->peer;
    PyObject* attr;

    char* name = PyString_AS_STRING(attr_name);
    if (0 == strcmp(name, "roundtriptime"))
    {
        return PyInt_FromLong((long)peer->roundTripTime);
    }
    else if (0 == strcmp(name, "packetloss"))
    {
        return PyInt_FromLong(peer->packetLoss);
    }

    attr = PyObject_GenericGetAttr((PyObject*)self, attr_name);
    // :TODO: Should below be Py_DECREF???
    if (NULL != attr)
        Py_INCREF(attr);
    return attr ;
}

static PyMethodDef enet_PeerType_methods[] = {
    {"send", (PyCFunction)enet_PeerObject_send, METH_VARARGS, enet_PeerObject_send_doc},
    {"receive", (PyCFunction)enet_PeerObject_receive, METH_VARARGS, enet_PeerObject_receive_doc},
    {"ping", (PyCFunction)enet_PeerObject_ping, METH_NOARGS, enet_PeerObject_ping_doc},
    {"reset", (PyCFunction)enet_PeerObject_reset, METH_NOARGS, enet_PeerObject_reset_doc},
    {"disconnect", (PyCFunction)enet_PeerObject_disconnect, METH_NOARGS, enet_PeerObject_disconnect_doc},
    {"throttle_configure", (PyCFunction)enet_PeerObject_throttle_configure, METH_VARARGS, enet_PeerObject_throttle_configure_doc},
    {NULL, NULL, 0, NULL} /* sentinel */
};

static PyTypeObject enet_PeerType = {
    /* The ob_type field must be initialized in the module init function
     * to be portable to Windows without using C++. */
    PyObject_HEAD_INIT(NULL)
    0,                          /*ob_size*/
    "Peer",                     /*tp_name*/
    sizeof(enet_PeerType),      /*tp_basicsize*/
    0,                          /*tp_itemsize*/
    /* Methods to implement standard operations */
    enet_PeerObjectDealloc,     /*tp_dealloc*/
    0,                          /*tp_print*/
    0,                          /*tp_getattr*/
    0,                          /*tp_setattr*/
    0,                          /*tp_compare*/
    0,                          /*tp_repr*/
    /* Method suites for standard classes */
    0,                          /*tp_as_number*/
    0,                          /*tp_as_sequence*/
    0,                          /*tp_as_mapping*/
    /* More standard operations (here for binary compatibility) */
    0,                          /*tp_hash*/
    0,                          /*tp_call*/
    0,                          /*tp_str*/
    (PyCFunction)enet_PeerObject_GetAttr, /*tp_getattro*/
    PyObject_GenericSetAttr,    /*tp_setattro*/
     /* Functions to access object as input/output buffer */
    0,                          /*tp_as_buffer*/
    /* Flags to define presence of optional/expanded features */
    Py_TPFLAGS_DEFAULT,         /*tp_flags*/
    0,                          /*tp_doc*/
    /* Assigned meaning in release 2.0 */
    /* call function for all accessible objects */
    0,                          /*tp_traverse*/
    /* delete references to contained objects */
    0,                          /*tp_clear*/
    /* Assigned meaning in release 2.1 */
    /* rich comparisons */
    0,                          /*tp_richcompare*/
    /* weak reference enabler */
    0,                          /*tp_weaklistoffset*/
    /* Added in release 2.2 */
    /* Iterators */
    0,                          /*tp_iter*/
    0,                          /*tp_iternext*/
    /* Attribute descriptor and subclassing stuff */
    enet_PeerType_methods,      /*tp_methods*/
    0,                          /*tp_members*/
    0,                          /*tp_getset*/
    0,                          /*tp_base*/
    0,                          /*tp_dict*/
    0,                          /*tp_descr_get*/
    0,                          /*tp_descr_set*/
    0,                          /*tp_dictoffset*/
    0,                          /*tp_init*/
    0,                          /*tp_alloc*/
    0,                          /*tp_new*/
    0,                          /*tp_free*/
    0,                          /*tp_is_gc*/
    0,                          /*tp_bases*/
    0,                          /*tp_mro*/
    0,                          /*tp_cache*/
    0,                          /*tp_subclasses*/
    0,                          /*tp_weaklist*/
};

static PyMethodDef enet_methods[] = {
	{"host", enet_HostObjectNew, METH_VARARGS, enet_HostObjectNew_doc}, /* Random test */
	{NULL, NULL} /* Sentinel */
};

void initenet(void)
{
    PyObject *module, *dict;
    PyObject *tmp;

    enet_HostType.ob_type = &PyType_Type;
    if (PyType_Ready(&enet_HostType))
        return;
    
    enet_PeerType.ob_type = &PyType_Type;
    if (PyType_Ready(&enet_PeerType))
        return;

    enet_EventType.ob_type = &PyType_Type;
    if (PyType_Ready(&enet_EventType))
        return;

    /* Initialize ENet library */
    if (-1 == enet_initialize())
        return;

	module = Py_InitModule3("enet", enet_methods, enet_doc);
    dict = PyModule_GetDict(module);
    tmp = PyInt_FromLong((long)ENET_EVENT_TYPE_NONE);
    PyDict_SetItemString(dict, "EVENT_NONE", tmp);
    Py_DECREF(tmp);
    tmp = PyInt_FromLong((long)ENET_EVENT_TYPE_CONNECT);
    PyDict_SetItemString(dict, "EVENT_CONNECT", tmp);
    Py_DECREF(tmp);
    tmp = PyInt_FromLong((long)ENET_EVENT_TYPE_DISCONNECT);
    PyDict_SetItemString(dict, "EVENT_DISCONNECT", tmp);
    Py_DECREF(tmp);
    tmp = PyInt_FromLong((long)ENET_EVENT_TYPE_RECEIVE);
    PyDict_SetItemString(dict, "EVENT_RECEIVE", tmp);
    Py_DECREF(tmp);
    tmp = PyInt_FromLong((long)ENET_PACKET_FLAG_RELIABLE);
    PyDict_SetItemString(dict, "FLAG_RELIABLE", tmp);
    Py_DECREF(tmp);
}
